# Router.rb
# A simple rubberband topological router based on the region concept mentioned in
# Tal Dayan's PhD thesis and other papers. 
# Currently we use random points as terminals and try to connect pairs of them.
# The initial delaunay triangulation is created by the gts library.
# Currently we only generate a PGN picture (lines and arcs) using cairo.
#
# Call: ruby router.rb
# or ruby router.rb seed
# for reproduceable results
#
require 'priority_queue'
require_relative 'gts'
require_relative 'canvas'

Board_Size = 800 # cairo PNG picture
Points = 64 # terminal count
Pin_Radius = 10
Trace_Width = 7
Clearance = 4
MinCutSize = 50

def init_seed
	seed = (ARGV[0] ? ARGV[0].to_i : rand(1000))
	print 'seed is ', seed, "\n"
	srand(seed)
end

	def inner_angle(o, a, b)
		ax = a.x - o.x
		ay = a.y - o.y
		bx = b.x - o.x
		by = b.y - o.y
		d = (Math.atan2(by, bx) - Math.atan2(ay, ax)).abs
		d =  2 * Math::PI - d if d > Math::PI
		d
	end

# Net connecting two Terminals (2Net)
class NetDesc
	@@id = 0
	attr_accessor :name1, :name2 # name of terminals
	attr_accessor :thickness
	attr_accessor :separation
	attr_accessor :id
	attr_accessor :flag # used for sorting
	#attr_accessor :score # for sorting attached nets of a terminal by angle 
	def initialize(name1, name2, thickness = Trace_Width, separation = Clearance)
	  @id = @@id
		@@id += 1
		@name1, @name2 = name1, name2
		@thickness, @separation = thickness, separation
	end
end

class NetDescList < Array
end

# Incident or attached net of a terminal
class Step
	attr_accessor :vertex # the terminal we are attached or incident to
	attr_accessor :index
	attr_accessor :score # for sorting attached nets of a terminal by angle 
	attr_accessor :ref
	attr_accessor :prev, :next # terminal
	attr_accessor :pstep, :nstep # step
	attr_accessor :id # id of this net
	attr_accessor :net_desc
	attr_accessor :lft, :rgt
	attr_accessor :radius
	def initialize(prev, nxt, id)
		@prev, @next, @id = prev, nxt, id
		@radius = 0 # default for incident net
	end
end

# Terminal
class Vertex
	attr_accessor :x, :y # position
	attr_accessor :iid
	attr_accessor :radius # outer copper
	attr_accessor :separation # outer clearance
	attr_accessor :neighbors # Vertex/Terminal, the neighbors in the delaunay triangulation
	attr_accessor :out # may be obsolete
	attr_accessor :incident_nets # Step, nets incident to this terminal
	attr_accessor :attached_nets # Step, nets atached to this terminal
	attr_accessor :name # name of the Terminal, i.e. U7_3 -- pin 3 of IC U7
	attr_accessor :tradius, :trgt
  def initialize(x = 0, y = 0, z = 0)
		@radius, @separation = Pin_Radius, Clearance
		@name = ''
		@x = x
		@y = y
		@iid = z
		@neighbors = Array.new
		@incident_nets = Array.new
		@attached_nets = Array.new
	end

  def reset_initial_size
		@radius, @separation = Pin_Radius, Clearance
	end

	def resize 
		reset_initial_size
		attached_nets.each{|step|
			net = step.net_desc
			trace_sep = [separation, net.separation].max
			@radius += trace_sep + net.thickness
			step.radius = @radius - net.thickness * 0.5
			@separation = net.separation
		}
	end

  def update(s)
		net = s.net_desc
		#fail unless net
		trace_sep = [@separation, net.separation].max
		@radius += trace_sep + net.thickness
		s.radius = @radius - net.thickness * 0.5
		@separation = net.separation
		#@attached_nets << s
	end

	def hash
    [@x, @y].hash
  end

  def eql?(other)
    [@x, @y].eql?([other.x, other.y])
  end

	# may be obsolete
  def rgt(id)
		#incident_nets.each{|s| return s if s.id == id}
 		incident_nets.each{|s| return false if s.id == id}
 		attached_nets.each{|s| return s.rgt if s.id == id}
		return true
	end

	# returns step
  def net(id)
		incident_nets.each{|s| return s if s.id == id}
 		attached_nets.each{|s| return s if s.id == id}
		return nil
	end

  # delete step
	def delete_net(id)
		#net = net(id)
		#d = net.net_desc.thickness# + net.net_desc.separation
		#net.vertex.radius -= d
		incident_nets.delete_if{|step| step.id == id}
		attached_nets.delete_if{|step| step.id == id}
		resize
	end

	def sort_attached_nets # by angle of total trace
		def _full_angle(o, a, b)
			return -1 unless a && b # > 2*PI, so we sort ending (incident) nets to outer most position  
			d = Math.atan2(b.y - o.y, b.x - o.x) - Math.atan2(a.y - o.y, a.x - o.x)
			#d < 0 ? d + 2 * Math::PI : d
			d < 0 ? 2 * Math::PI + d  : d
		end
		def _cross(o, a, b)
			(a.x - o.x) * (b.y - o.y) - (a.y - o.y) * (b.x - o.x)
		end
		unless attached_nets.length < 2
			attached_nets.each{|n| fail unless n.vertex == self;n.index = -inner_angle(self, n.next, n.prev)} # largest angle outer most
			attached_nets.sort_by!{|n| n.index}
			attached_nets.each_with_index{|n, i| n.index = i}
			shash = Hash.new
			attached_nets.each{|n| # group attached nets with same angle (overlapping)
				l = n.prev
				r = n.next
				n.net_desc.flag = 1
				if shash.has_key?([l, r])
					shash[[l, r]] << n
				elsif shash.has_key?([r, l])
					n.net_desc.flag = -1 # inverted direction
					shash[[r, l]] << n
				else
					shash[[l, r]] = [n]
				end
			}
			shash.each_value{|group| # fine sort each group by following the traces as long as they are parallel
				if group.length > 1
					group.each{|el| el.ref = el}
					group_sign = _cross(group[0].vertex, group[0].prev, group[0].next) <=> 0
					gr = group.dup
					indices = Array.new
					rel = Hash.new(0)
					rul = Hash.new(0)
					gr.each{|el| indices << el.index}
					indices.sort!
					while gr.length > 1
						gr.map!{|el| (el.net_desc.flag == group_sign ? el.pstep : el.nstep)} # walk in one direction
						gr.keep_if{|el| el.next && el.prev}
						gr.each{|el| el.ref = (el.net_desc.flag == group_sign ? el.nstep.ref : el.pstep.ref)} 
						gr.each{|el| el.score = (el.net_desc.flag != group_sign ? _full_angle(el.vertex, el.prev, el.next) : _full_angle(el.vertex, el.next, el.prev))}
						gr.combination(2).each{|el|
							a, b = *el
							puts a.score, b.score
							c = (a.score <=> b.score)
							puts c
							if c != 0 and !rel.has_key?([a.ref, b.ref])
								rel[[a.ref, b.ref]] = c
								rel[[b.ref, a.ref]] = -c
							end
						}
					end
					gr = group.dup
					group_sign *= -1
					while gr.length > 1
						gr.map!{|el| (el.net_desc.flag == group_sign ? el.pstep : el.nstep)} # walk in one direction
						gr.keep_if{|el| el.next && el.prev}
						gr.each{|el| el.ref = (el.net_desc.flag == group_sign ? el.nstep.ref : el.pstep.ref)} 
						gr.each{|el| el.score = (el.net_desc.flag != -group_sign ? _full_angle(el.vertex, el.prev, el.next) : _full_angle(el.vertex, el.next, el.prev))}
						gr.combination(2).each{|el|
							a, b = *el
							puts a.score, b.score
							c = (a.score <=> b.score)
							puts c

							if c != 0 and !rul.has_key?([a.ref, b.ref])
								rul[[a.ref, b.ref]] = c
								rul[[b.ref, a.ref]] = -c
							end
						}
					end
					rel.each_pair{|k, v|
						if rul.has_key?(k) and (rul[k] != v)
							puts k, v, rul[k] # contradictions may occur :-(
							#fail
						end
					}
					rel.merge!(rul){|k, v1, v2| 0} # dirty hack
					group.sort!{|a, b| rel[[a, b]]}
					group.each{|el| el.index = indices.shift}
				end
			}
			attached_nets.sort_by!{|el| el.index}
		end
	end
end

class Region
	attr_accessor :vertex, :neighbors, :name, :incident
  def initialize(v = nil)
		@vertex = v
		@neighbors = Array.new
		@name = nil
		@incident = true
	end
end

class Cut
  attr_accessor :cap, :flow
  def initialize(dx, dy)
		@cap = Math.sqrt(dx ** 2 + dy ** 2)
		@flow = 0
	end
end

# key is [ax, ay, bx, by] -- coordinates of two points
# should be insensitive when points are exchanged.
class Cut_Hash < Hash
  def [](k0, k1, k2, k3)
		if k0 < k2
			super([k0, k1, k2, k3])
		elsif k0 == k2
			k1 > k3 ? super([k2, k3, k0, k1]): super([k0, k1, k2, k3])
	  else
			super([k2, k3, k0, k1])
    end
  end

  def []=(k0, k1, k2, k3, v)
		if k0 < k2
			super([k0, k1, k2, k3], v)
		elsif k0 == k2
			k1 > k3 ? super([k2, k3, k0, k1], v): super([k0, k1, k2, k3], v)
	  else
			super([k2, k3, k0, k1], v)
    end
  end
end


class Router
  def initialize
	  @Path_ID = 0
		@image = Cairo::ImageSurface.new(Board_Size, Board_Size)
		@pic = Canvas::Pic.new(@image)
		@pic.set_source_rgba(0.8, 0.8, 0.8, 1)
		@pic.paint
		s = Gts::Surface.new
		pa = Gts::Point.new(0, 0, 0)
		pb = Gts::Point.new(0, Board_Size, 0)
		pc = Gts::Point.new(Board_Size, 0, 0)
		pd = Gts::Point.new(Board_Size, Board_Size, 0)
		t = Gts::Triangle.triangle_enclosing([pa, pb, pc, pd], 2) 
		f = Gts::Face.new(t.e1, t.e2, t.e3)
		s.add_face(f)

va = Gts::Vertex.new(0, 0, 0)
vb = Gts::Vertex.new(0, Board_Size, 1)
vc = Gts::Vertex.new(Board_Size, 0, 2)
vd = Gts::Vertex.new(Board_Size, Board_Size, 3)
s.delaunay_add_vertex(va)
s.delaunay_add_vertex(vb)
s.delaunay_add_vertex(vc)

s.delaunay_add_vertex(vd)

		cell = Hash.new

cell[[0 / 30, 0 / 30]] = 1
cell[[0 / 30, Board_Size / 30]] = 1
cell[[Board_Size / 30, 0 / 30]] = 1
cell[[Board_Size / 30, Board_Size / 30]] = 1

v5 = Gts::Vertex.new(300, 300, 4)
v6 = Gts::Vertex.new(300, 500, 5)
v7 = Gts::Vertex.new(500, 300, 6)
v8 = Gts::Vertex.new(500, 500, 7)

cell[[300 / 30, 300 / 30]] = 1
cell[[300 / 30, 500 / 30]] = 1
cell[[500 / 30, 300 / 30]] = 1
cell[[500 / 30, 500 / 30]] = 1

		id = 8
		while id < Points
		r1, r2 = rand(Board_Size-20)+10, rand(Board_Size-20)+10
		if cell[[r1 / 30, r2 / 30]] == nil
		#if true#unless (300..500).include?(r1) or (300..500).include?(r2)
			v = Gts::Vertex.new(r1, r2, id)
			s.delaunay_add_vertex(v)
			id += 1
		end
cell[[r1 / 30, r2 / 30]] = 1
		end

s.delaunay_add_vertex(v5)
s.delaunay_add_vertex(v6)

c = Gts::Constraint.new(v5, v6)
puts s.delaunay_add_constraint(c)

s.delaunay_add_vertex(v7)

c = Gts::Constraint.new(v5, v7)
puts s.delaunay_add_constraint(c)

s.delaunay_add_vertex(v8)

c = Gts::Constraint.new(v8, v7)
puts s.delaunay_add_constraint(c)

c = Gts::Constraint.new(v8, v6)
puts s.delaunay_add_constraint(c)

#c = Gts::Constraint.new(v5, v8)
#puts s.delaunay_add_constraint(c)

#puts s.delaunay_remove_holes

		t.destroy_enclosing
		@vertices = Array.new#(Points)
		@regions = Array.new#(Points)
	#	s.uniq!
		s.each_vertex{|gts_v|
			i = gts_v.z.round
			@vertices[i] = v = Vertex.new(gts_v.x, gts_v.y, i)
			@regions[i] = Region.new(v)
		}

		#set = @vertices.sample(2)

    #set = (0..9).to_a.map{|el| @vertices.sample(2)}
		#tag = true
		set = @vertices.select{|el| el.iid > 3}.each_slice(2).to_a.shuffle
		#set = @vertices.partition{tag = !tag}.transpose 
		set.sort_by!{|el| (el[0].x - el[1].x) ** 2 + (el[0].y - el[1].y) ** 2}

		@netlist = NetDescList.new
		#9.times{|i|
		(0..9).each{|i|

			#v1 = @vertices[rand(Points)]
			#v2 = @vertices[rand(Points)]
			v1, v2 = *set[i*3]
			v1.name = i.to_s + 's'
			v2.name = i.to_s + 'e'
			net_desc = NetDesc.new(v1.name, v2.name)
			#net_desc.thickness = 20 - 16
			@netlist << net_desc
		}

		@cuts = Cut_Hash.new
		s.each_vertex{|gts_v|
			i = gts_v.z.round
			gts_v.neighbors(s).each{|n|
				j = n.z.round
				@vertices[i].neighbors << @vertices[j]
				@regions[i].neighbors << @regions[j]
				ax = @vertices[i].x
				ay = @vertices[i].y
				bx = @vertices[j].x
				by = @vertices[j].y
				@cuts[ax, ay, bx, by] = Cut.new(ax - bx, ay - by)
			}
		}
	end
	def draw_vertices
		@pic.set_source_rgba(0, 0, 0, 0.3)
		@pic.set_line_width(1)
		@vertices.each{|v|
				@pic.new_sub_path
				@pic.set_line_width(1)
				@pic.arc(v.x, v.y, Pin_Radius, 0, 2*Math::PI)
				@pic.fill
				#@pic.stroke

@pic.stroke



			v.neighbors.each{|n|
				@pic.move_to(v.x, v.y); @pic.line_to(n.x, n.y)
@pic.stroke



			}
		}
		@pic.stroke
		@pic.set_source_rgba(1, 0, 0, 0.7)
		@pic.set_line_width(2)

		@cuts.each_pair{|k, v|
			if v.cap < MinCutSize
				@pic.move_to(k[0], k[1])
				@pic.line_to(k[2], k[3])
			end
		}
		@pic.stroke

	end

	def gen_line(x0, y0, x1, y1, w)
		@pic.set_line_width(w)
		@pic.move_to(x0, y0)
		@pic.line_to(x1, y1)
		@pic.stroke
	end

  def gen_arc(x, y, r, from, to, width)
		@pic.new_sub_path
		@pic.set_line_width(width)
		@pic.arc(x, y, r, from, to)
		@pic.stroke
	end

# http://www.faqs.org/faqs/graphics/algorithms-faq/
# http://www.ecse.rpi.edu/Homepages/wrf/Research/Short_Notes/pnpoly.html
# int pnpoly(int nvert, float *vertx, float *verty, float testx, float testy)
# {
#   int i, j, c = 0;
#   for (i = 0, j = nvert-1; i < nvert; j = i++) {
#     if ( ((verty[i]>testy) != (verty[j]>testy)) &&
# 	 (testx < (vertx[j]-vertx[i]) * (testy-verty[i]) / (verty[j]-verty[i]) + vertx[i]) )
#        c = !c;
#   }
#   return c;
# }
#
# input: array of vertices
# result: the vertices inside the polygon (or on the border?)
	def novertices_in_polygon(p_vertices, test_vertices)
		res = Array.new
		nvert = p_vertices.length
		test_vertices.each_with_index{|tp, k|
			tx = tp.x
			ty = tp.y
			i = -1
			j = nvert - 1
			c = false
			while (i+=1) < nvert
        if ((((p_vertices[i].y<=ty) && (ty<p_vertices[j].y)) ||
             ((p_vertices[j].y<=ty) && (ty<p_vertices[i].y))) &&
            (tx < (p_vertices[j].x - p_vertices[i].x) * (ty - p_vertices[i].y) / (p_vertices[j].y - p_vertices[i].y) + p_vertices[i].x))



#				if (((p_vertices[i].y > testy) != (p_vertices[j].y > testy)) &&
#					(testx < (p_vertices[j].x - p_vertices[i].x) * (testy - p_vertices[i].y) / (p_vertices[j].y - p_vertices[i].y) + p_vertices[i].x))
					 c = !c
				end
				j = i
				
			end
			if c 
#puts 'yxc', tp
res << test_vertices[k]
			end
			#(res << tp) if c
		}
		#p res
		return res
		#[]
	end

	def vertices_in_polygon(p_vertices, test_vertices)
		res = Array.new
		nm1 = p_vertices.length - 1
		test_vertices.each_with_index{|tp, k|
			ty = tp.y
			i = 0
			j = nm1
			c = false
			while i <= nm1
        if ((((p_vertices[i].y <= ty) && (ty < p_vertices[j].y)) ||
             ((p_vertices[j].y <= ty) && (ty < p_vertices[i].y))) &&
            (tp.x < (p_vertices[j].x - p_vertices[i].x) * (ty - p_vertices[i].y) / (p_vertices[j].y - p_vertices[i].y) + p_vertices[i].x))
					 c = !c
				end
				j = i
				i += 1
			end
			res << tp if c
		}
		res
	end


	#      (c)
	#     /
	#    /     (p)
	#   /
	# (b)
	# see http://www.geometrictools.com/
	#
	def distance_line_segment_point_squared(bx, by, cx, cy, px, py)
		mx = cx - bx
		my = cy - by
		hx = px - bx
		hy = py - by
		t0 = (mx * hx + my * hy).fdiv(mx ** 2 + my ** 2)
		if t0 <= 0
		elsif t0 < 1
			hx -= t0 * mx
			hy -= t0 * my
		else
			hx -= mx
			hy -= my
		end
		return hx ** 2 + hy ** 2
	end

	def vertices_touch_line(x1, y1, x2, y2, line_thickness, line_separation, vertices)
		v_in = Array.new
		vertices.each{|v|
			s = [v.separation, line_separation].max
			if distance_line_segment_point_squared(x1, y1, x2, y2, v.x, v.y) < (v.radius + s + line_thickness * 0.5) ** 2
				v_in << v
			end
		}
		v_in
	end

	# P_IN = 1; P_ON = 0; P_OUT = -1 
	# see http://en.wikipedia.org/wiki/Barycentric_coordinates_%28mathematics%29
	def unused_point_in_triangle(x1, y1, x2, y2, x3, y3, x, y)
		d  =  (y2 - y3) * (x1 - x3) + (x3 - x2) * (y1 - y3)
		l1 = ((y2 - y3) * (x - x3) + (x3 - x2) * (y - y3)) / d
		l2 = ((y3 - y1) * (x - x3) + (x1 - x3) * (y - y3)) / d
		l3 = 1 - l1 - l2
		min, max = [l1, l2, l3].minmax
		if 0 <= min && max <= 1
			0 < min && max < 1 ? P_IN : P_ON
		else
			P_OUT
		end
	end

	def vertices_in_triangle(x1, y1, x2, y2, x3, y3, vertices)
		d  =  (y2 - y3) * (x1 - x3) + (x3 - x2) * (y1 - y3)
		v_in = Array.new
		if d == 0
			puts 'vertices_in_triangle, degenerated'
			return v_in
		end
		y2my3 = (y2 - y3) / d
		x3mx2 = (x3 - x2) / d
		x1mx3 = (x1 - x3) / d
		y3my1 = (y3 - y1) / d
		vertices.each{|v|
			vxmx3 = v.x - x3
			vymy3 = v.y - y3
			l1 = y2my3 * vxmx3 + x3mx2 * vymy3
			l2 = y3my1 * vxmx3 + x1mx3 * vymy3
			min, max = [l1, l2, 1 - l1 - l2].minmax
			if 0 <= min && max <= 1
				v_in << v
			end
		}
		v_in
	end

	# see http://en.wikibooks.org/wiki/Algorithm_Implementation/Geometry/Convex_hull/Monotone_chain
	def unused_convex_hull(points)
		points.sort!.uniq!
		return points if points.length < 3
		def hcross(o, a, b)
			(a[0] - o[0]) * (b[1] - o[1]) - (a[1] - o[1]) * (b[0] - o[0])
		end
		lower = Array.new
		points.each{|p|
			while lower.length > 1 and hcross(lower[-2], lower[-1], p) <= 0 do lower.pop end
			lower.push(p)
		}
		upper = Array.new
		points.reverse_each{|p|
			while upper.length > 1 and hcross(upper[-2], upper[-1], p) <= 0 do upper.pop end
			upper.push(p)
		}
		return lower[0...-1] + upper[0...-1]
	end

	def pconvex_vertices(vertices, h1, h2)
	return vertices if vertices.length < 2

		cx = vertices.inject(0){|sum, el| sum + el.x} / vertices.length
		cy = vertices.inject(0){|sum, el| sum + el.y} / vertices.length
		vertices.sort_by!{|el| Math.atan2(el.y - cy, el.x - cx)}
		i = vertices.length-1
		a = vertices[-1]
		h = Hash.new
		while (i -= 1) >= -1
		puts '***'
			b = vertices[i]
			fail if a == b
			bx, by, ax, ay = *get_tangents(b.x, b.y, b.tradius, b.trgt, a.x, a.y, a.tradius, a.trgt)
			gen_line(ax, ay, bx, by,1)
			h[[ax, ay]] = a
			h[[bx, by]] = b
			a = b
		end
		(unused_convex_hull(h.keys + [h1, h2])-[h1, h2]).map{|el| h[el]}
	end


	def convex_vertices(vertices, prev, nxt, h1, h2)
		#puts 'convex_vertices'
		vertices.delete(prev)
		vertices.delete(nxt)
		return vertices if vertices.length < 1
		h = Hash.new
if vertices.length > 1
		cx = vertices.inject(0){|sum, el| sum + el.x} / vertices.length
		cy = vertices.inject(0){|sum, el| sum + el.y} / vertices.length
		vertices.sort_by!{|el| Math.atan2(el.y - cy, el.x - cx)}
		i = vertices.length
		a = vertices[0]
		#while (i -= 1) > 0
		while i > 0
		i -= 1
		#puts '***', i, '---'
			b = vertices[i]
			fail if a == b
			#atrgt = a.trgt
			#btrgt = b.trgt
			#if a == prev then atrgt = !atrgt end
			#if b == nxt then btrgt = !btrgt end
			bx, by, ax, ay = *get_tangents(b.x, b.y, b.tradius, true, a.x, a.y, a.tradius, true)
			#gen_line(ax, ay, bx, by,1)
			h[[ax, ay]] = a
			h[[bx, by]] = b
			a = b
		end
end
		vertices.each{|el|
			bx, by, ax, ay = *get_tangents(prev.x, prev.y, prev.tradius, prev.trgt, el.x, el.y, el.tradius, el.trgt)
			#gen_line(ax, ay, bx, by,1)
			h[[ax, ay]] = el
			h[[bx, by]] = prev
			bx, by, ax, ay = *get_tangents(el.x, el.y, el.tradius, el.trgt, nxt.x, nxt.y, nxt.tradius, nxt.trgt)
			#gen_line(ax, ay, bx, by,1)
			h[[ax, ay]] = nxt
			h[[bx, by]] = el
		}
		#return []
		#puts vertices
		#puts 'yyy'
		return (((unused_convex_hull(h.keys + [h1, h2])-[h1, h2]).map{|el| h[el]}).uniq) #or [])
		#return vertices
	end











	def incident_net_to_neighbors?(vertex, neighbors)
		!(vertex.incident_nets.map{|el| el.next || el.prev} & neighbors).empty?
	end

	def unused_basic_dijkstra(start_node)
		# Nodes that may have neighbours wich can be relaxed further
		active = PriorityQueue.new         
		# Best distances found so far
		distances = Hash.new { 1.0 / 0.0 } 
		# Parent pointers describing shortest paths for all nodes
		parents = Hash.new                 
		# Initialize with start node
		active[start_node] = 0
		until active.empty?
			u, distance = active.delete_min
			distances[u] = distance
			d = distance + 1
			u.neighbours.each do | v |
				next unless d < distances[v] # we can't relax this one
				active[v] = distances[v] = d
				parents[v] = u
			end    
		end
		parents
	end
	
	def dijkstra(start_node, end_node_name)
		q = PriorityQueue.new
		distances = Hash.new(Float::INFINITY)
		parents = Hash.new
		fail unless start_node
		q[[start_node, false]] = 0
		#cross = last_cross = false
		cross = false
		while true do
			min, distance = q.delete_min
			u, last_cross = *min
			#puts 'ddd', min, u
			if !min or ((u.vertex.name == end_node_name) and (u.incident) and parents[parents[[u, last_cross]]])
				#parents[end_node_name] = min
				break
			end
			distances[min] = distance
			x = u.vertex.x
			y = u.vertex.y
			u.neighbors.each do | v |
			fail unless v
				d = distance + Math.sqrt((v.vertex.x - x) ** 2 + (v.vertex.y - y) ** 2)
				old = parents[[u, last_cross]]
				old = old[0] if old
				if old
					cross = no_cross(old, u, v) > 0
					next unless d < distances[[v, cross]]
					lcuts = bor_list(old, v, u)
					lcuts << old.vertex if ((last_cross != nil) and (cross != last_cross))
					lcap = 1e3
					lcuts.each{|c|
						lcap = [@cuts[u.vertex.x, u.vertex.y, c.x, c.y].cap, lcap].min
					}
					hhh = lcuts# + [old, v]
					lcap = 0 if incident_net_to_neighbors?(u.vertex, lcuts)
					lcap = 0 if old == v
					jki = u.vertex.incident_nets.select{|el| el.next == old.vertex ||  el.prev == old.vertex}
					#lcap = 0 unless jki.empty?
					jki.each{|el|
						lcap = 0 if cross != last_cross
						if el.next
							n = el.next
							nn = el.nstep.next
						else
							n = el.prev
							nn = el.pstep.prev
						end
						lcap = 0 if nn && (cross(u.vertex, n, nn) > 0) != cross
					}
					jki = u.vertex.incident_nets.select{|el| el.next == v.vertex ||  el.prev == v.vertex}
					#lcap = 0 unless jki.empty?
					jki.each{|el|
					lcap = 0 if cross != last_cross
						if el.next
							n = el.next
							nn = el.nstep.next
						else
						#lcap = 0
							n = el.prev
							nn = el.pstep.prev
						end
						lcap = 0 if nn && (cross(u.vertex, n, nn) > 0) == cross # condition should be ok
					}
					jki = old.vertex.incident_nets.select{|el| el.next == u.vertex ||  el.prev == u.vertex}
					lcap = 0 unless jki.empty? or (cross == last_cross)
					next if lcap < MinCutSize
				end
				q[[v, cross]] = distances[[v, cross]] = d
				parents[[v, cross]] = [u, last_cross]
			end    
		end
		return nil unless min
		purents = Hash.new
		purents[end_node_name] = u 
		p = min
		while p
			n =  parents[p]
		  purents[p[0]] = n && n[0]
			p = n
		end
		return purents
	end


	#          /neighbor
	#     a   /
	# <-------\
	#       /  \b
	#      /    \
	#return these
	def no_split_neighbor_list(a, b, n)
		aax = n.vertex.x - a.vertex.x
		aay = n.vertex.y - a.vertex.y
		bbx = b.vertex.x - n.vertex.x
		bby = b.vertex.y - n.vertex.y
		as = aay.fdiv(aax)
		bs = bby.fdiv(bbx)
		l = Array.new
		n.neighbors.each{|el|
		if aax*bby-aay*bbx>0
			if (aax * (el.vertex.y - n.vertex.y) - aay * (el.vertex.x - n.vertex.x) > 0) and (bbx * (el.vertex.y - n.vertex.y) - bby * (el.vertex.x - n.vertex.x) > 0)
				l << el
			end
		else
			if (aax * (el.vertex.y - n.vertex.y) - aay * (el.vertex.x - n.vertex.x) > 0) or (bbx * (el.vertex.y - n.vertex.y) - bby * (el.vertex.x - n.vertex.x) > 0)
				l << el
			end

		end
		}
		l.delete(a)
		l.delete(b)
		return l
	end

	def no_cross(o, a, b)
		(a.vertex.x - o.vertex.x) * (b.vertex.y - o.vertex.y) - (a.vertex.y - o.vertex.y) * (b.vertex.x - o.vertex.x)
	end


	def inner_angle(o, a, b)
		ax = a.x - o.x
		ay = a.y - o.y
		bx = b.x - o.x
		by = b.y - o.y
		d = (Math.atan2(by, bx) - Math.atan2(ay, ax)).abs
		d =  2 * Math::PI - d if d > Math::PI
		d
	end

	def cross(o, a, b)
		(a.x - o.x) * (b.y - o.y) - (a.y - o.y) * (b.x - o.x)
	end

	def cross_tangents(a, b, c, id)
		last_step, cur_step, nxt_step = a.net(id), b.net(id), c.net(id)
		t1 = get_tangents(a.x, a.y, last_step.radius, last_step.rgt, b.x, b.y, cur_step.radius, cur_step.rgt)
		t2 = get_tangents(b.x, b.y, cur_step.radius, cur_step.rgt, c.x, c.y, nxt_step.radius, nxt_step.rgt)
		((t1[2] - t1[0]) * (t2[3] - t2[1]) - (t1[3] - t1[1]) * (t2[2] - t2[0])) > 0
	end

	def bor_list(a, b, n)
		a, b, n = a.vertex, b.vertex, n.vertex
		aax = n.x - a.x
		aay = n.y - a.y
		bbx = b.x - n.x
		bby = b.y - n.y
		l = Array.new
		n.neighbors.each{|el|
		el_x_a = aax * (el.y - n.y) - aay * (el.x - n.x)
		el_x_b = bbx * (el.y - n.y) - bby * (el.x - n.x)
		if aax*bby-aay*bbx>0
			if el_x_a > 0 && el_x_b > 0
				l << el
			end
		else
			if el_x_a < 0 && el_x_b < 0
				l << el
			end
		end
		}
		l.delete(a)
		l.delete(b)
		return l
	end


	def old_bor_list(a, b, n)
		aax = n.vertex.x - a.vertex.x
		aay = n.vertex.y - a.vertex.y
		bbx = b.vertex.x - n.vertex.x
		bby = b.vertex.y - n.vertex.y
		l = Array.new
		n.neighbors.each{|el|
		el_x_a = aax * (el.vertex.y - n.vertex.y) - aay * (el.vertex.x - n.vertex.x)
		el_x_b = bbx * (el.vertex.y - n.vertex.y) - bby * (el.vertex.x - n.vertex.x)
		if aax*bby-aay*bbx>0
			if el_x_a > 0 && el_x_b > 0
				l << el
			end
		else
			if el_x_a < 0 && el_x_b < 0
				l << el
			end
		end
		}
		l.delete(a)
		l.delete(b)
		return l
	end

	def route(net_id)
		net_desc = @netlist[net_id]
		from, to = net_desc.name1, net_desc.name2
		return nil unless from
		start_node = nil
    @regions.each{|r|
		if r.incident
		#print  r.vertex.name, '__', from, "\n"
	    if r.vertex.name == from
        start_node = r
				break
		  end
		end
		}
		#puts net_id
fail unless start_node
		parents = dijkstra(start_node, to)
		unless parents
puts 'dijkstra failed!'
return
		end
		r1 = r2 = nil
		cur = parents[to]
		prv = nil
		@pic.move_to(cur.vertex.x, cur.vertex.y)
		while cur do
			nxt = parents[cur]
			if prv && nxt
				ne = no_split_neighbor_list(prv, nxt, cur)
				cap = 1e3
				ne.each{|r|
					if (i = @cuts[cur.vertex.x, cur.vertex.y, r.vertex.x, r.vertex.y].cap) < cap then cap = i end
				}
				ne_comp = ((cur.neighbors - ne) - [prv, nxt])#.uniq
				ne << nxt
				ne_comp << nxt
				if r1
					ne.delete(r1)
					ne.delete(r2)
					ne_comp.delete(r1)
					ne_comp.delete(r2)
					ne << r1
					ne_comp << r2
				else
					ne << prv
					ne_comp << prv
				end
				@regions.delete(cur)
				r1 = Region.new(cur.vertex)
				r2 = Region.new(cur.vertex)
				r1.name='r1'
				r2.name='r2'
				@regions << r1 << r2
				cur.neighbors.each{|el|
					el.neighbors.delete(cur)
				}
				ne.each{|el|
					el.neighbors << r1
					r1.neighbors << el
				}
				ne_comp.each{|el|
					el.neighbors << r2
					r2.neighbors << el
				}
			end
			if prv && nxt
			if no_cross(prv, cur, nxt) > 0
				r1.incident = false
			else
				r2.incident = false
			end
			end
			#if out[cur]
			#	prev = parents[cur]
			#	puts 'out'
			#	if prev
			#		@pic.new_sub_path
			#		@pic.arc(prev.vertex.x, prev.vertex.y, 20, 0, 2 * Math::PI)
			#		@pic.stroke
			#	end
			#end
			prv && @pic.move_to(prv.vertex.x, prv.vertex.y)
			@pic.line_to(cur.vertex.x, cur.vertex.y)
			@pic.stroke
      prv = cur
			cur = nxt
		end
		@pic.stroke
		cur = prv = parents[to]
		p = Array.new
		while cur
			v = cur.vertex
			#v.out = out[prv]
			p << v
			prv = cur
			cur = parents[cur]
		end
		pa = Array.new
		pa << p[-1]
		i = p.length - 2
		while i > 0
			if cross(p[i - 1], p[i], p[i + 1])
				pa << p[i]
			else
				pa << p[i]
			end
			i -= 1
		end
		pa << p[0]
		path = Array.new
		pa.each_index{|i|  
			cur = pa[i]
			nxt = (i == pa.length - 1 ? nil : pa[i + 1])
			prv = (i == 0 ? nil : pa[i - 1])
			if prv and nxt
				path << pa[i]
			else
				path << pa[i]
			end
		}
		pstep = nil
		path.each_index{|i|  
			cur = path[i]
			nxt = (i == path.length - 1 ? nil : path[i + 1])
			prv = (i == 0 ? nil : path[i - 1])
			step = Step.new(prv , nxt, @Path_ID)
			step.net_desc = net_desc
			step.vertex = cur
			step.pstep = pstep
			pstep = step
			if prv and nxt
				cur.update(step)

				step.lft = (cross(prv, cur, nxt) > 0)
				#step.rgt = cur.out
step.rgt =	!step.lft
				cur.attached_nets << step
			else
				cur.incident_nets << step
			end
		}
		@Path_ID += 1

		s = pstep
		s.nstep = nil
		while p = s.pstep
			p.nstep = s
			s = p
		end
	end

	# see http://en.wikibooks.org/wiki/Algorithm_Implementation/Geometry/Tangents_between_two_circles
	def get_tangents(x1, y1, r1, l1, x2, y2, r2, l2)
		d_sq = (x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2)
		if (d_sq <= (r1-r2)*(r1-r2))
			puts x1, y1, r1, x2, y2, r2
			fail
		end
		d = Math.sqrt(d_sq)
		vx = (x2 - x1) / d
		vy = (y2 - y1) / d
		sign2 = sign1 = 1
		sign2 = (l2  ? -1 : 1)
		sign1 = (l1  ? -1 : 1)
		sign2 *= sign1
		c = (r1 - sign2 * r2) / d
		#if (c*c > 1.0) continue;
		h = Math.sqrt([0.0, 1.0 - c*c].max)
		nx = vx * c - sign1 * h * vy
		ny = vy * c + sign1 * h * vx
		a = Array.new(4)
		a[0] = x1 + r1 * nx
		a[1] = y1 + r1 * ny
		a[2] = x2 + sign2 * r2 * nx
		a[3] = y2 + sign2 * r2 * ny
		return a
	end

	def sym_get_tangents(x1, y1, r1, l1, x2, y2, r2, l2)
		d_sq = (x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2)
		if (d_sq <= (r1-r2)*(r1-r2)) then return nil end
		d = Math.sqrt(d_sq)
		vx = (x2 - x1) / d
		vy = (y2 - y1) / d
		sign2 = sign1 = 1
		sign2 = (l2  ? -1 : 1)
		sign1 = (l1  ? -1 : 1)
		sign2 *= sign1
		c = (r1 - sign2 * r2) / d
		#if (c*c > 1.0) continue;
		h = Math.sqrt([0.0, 1.0 - c*c].max)
		nx = vx * c - sign1 * h * vy
		ny = vy * c + sign1 * h * vx
		a = Array.new(4)
		a[0] = x1 + r1 * nx
		a[1] = y1 + r1 * ny
		a[2] = x2 + sign2 * r2 * nx
		a[3] = y2 + sign2 * r2 * ny
		return a
	end

	def replace(vertex, id, list)
		if list.empty?
			step = vertex.net(id)
 			step.next.net(id).prev = step.prev
 			step.prev.net(id).next = step.next
			step.pstep.nstep = step.nstep
			step.nstep.pstep = step.pstep
		else

		#puts 'replace_insert'
		  pstep = nil
			list.each_index{|i|
				n = Step.new(list[i - 1], list[i + 1], id)
				n.net_desc = @netlist[id]
				#puts 'id', id
				fail unless n.net_desc
				n.vertex = list[i]
				n.pstep = pstep
				pstep = n
				n.rgt = !vertex.net(id).rgt
				if i == 0
					n.prev = vertex.net(id).prev
					n.prev.net(id).next = list[0]
					n.pstep = vertex.net(id).pstep
					#n.prev.net(id).nstep = list[0]
				end
				if i == list.length - 1
					n.next = vertex.net(id).next
					n.next.net(id).prev = list[i]
					n.next.net(id).pstep = n
					n.nstep = vertex.net(id).nstep

				end
				#list[i].append_step(n)
				fail unless n.net_desc
				list[i].update(n)
				list[i].attached_nets << n
			}

		i = list.length
		s = pstep
		#s.nstep = nil
		while i > 0
		  i -= 1
		  p = s.pstep
			p.nstep = s
			s = p
		end

		end
		vertex.delete_net(id)
	end

	def init_vertices
		@vertices.each{|vert|
			vert.radius = Pin_Radius 
			vert.separation = Clearance 
		}
	end

	# sort attached nets and calculate its radii
	def no_prepare_steps
		@vertices.each{|vert|
			#vert.attached_nets.sort!{|a, b| inner_angle(a.prev, vert, a.next) <=> inner_angle(b.prev, vert, b.next)}
			#last_sep = vert.separation
			#sum = vert.radius
			
			vert.reset_initial_size
			vert.attached_nets.each{|step|

			net = step.net_desc

			trace_sep = [vert.separation, net.separation].max
			vert.radius += trace_sep + net.thickness
			step.radius = vert.radius - net.thickness * 0.5
			vert.separation = net.separation

			  #net = @netlist[step.id]
				#s = [net.separation, last_sep].max
				#step.radius = sum + s + net.thickness * 0.5
				#sum += s + net.thickness
				#last_sep = net.separation
			}
		}
	end

	def sort_attached_nets
		@vertices.each{|vert| vert.sort_attached_nets}
	end

  def nubly
	  replaced = true
		while replaced do
	  replaced = false

		@vertices.each{|vert|
			vert.attached_nets.reverse_each{|n|
			  id = n.id
				last = n.prev
				nxt = n.next
				cur = vert
				step = n
				if cross_tangents(last, cur, nxt, id) != step.rgt
				replaced = true
					#rep = vertices_in_triangle(last.x, last.y, cur.x, cur.y, nxt.x, nxt.y, @vertices)
					vec_x = nxt.x - last.x
					vec_y = nxt.y - last.y
					hx, hy = vec_x, vec_y
					vec_x, vec_y = -vec_y, vec_x
					if cross(last, nxt, cur) > 0
						vec_x, vec_y = -vec_x, -vec_y
					end
					h3 = Vertex.new(last.x + (hx + vec_x) * 0.5, last.y + (hy + vec_y) * 0.5)
					rep = vertices_in_polygon([last, cur, nxt, h3], @vertices) - [cur, h3]
					#h1 = Vertex.new(last.x + vec_x, last.y + vec_y)
					#h2 = Vertex.new(nxt.x + vec_x, nxt.y + vec_y)

					h1 = [last.x + vec_x, last.y + vec_y]
					h2 = [nxt.x + vec_x, nxt.y + vec_y]


					rep |= [last, nxt]
					#gen_arc(*h1, 15, 0, 6, 5)
					#gen_arc(*h2, 15, 0, 6, 5)
					#gen_arc(h3.x, h3.y, 15, 0, 6, 15)

					net = cur.net(id).net_desc
					rep.each{|v|
					if v == last || v == nxt
						s = v.net(id)
						v.trgt = s.rgt
						v.tradius = s.radius
					else
					#fail unless step ==  v.net(id)
						#s = v.net(id)
						v.trgt = !step.rgt
						v.tradius = v.radius + net.thickness * 0.5 + [net.separation, v.separation].max# + 10
						#print '/// ',  v.tradius, ' ',v.radius, ' ', v.separation, ' ', net.thickness, ' ', net.separation, "\n"
					end
					}
#rep.delete(cur)
set_color(1, 1, 0, 1)
#rep.each{|v| gen_arc(v.x, v.y, 10, 0, 5, 5)}
set_color(1, 1, 1, 1)

#rep.uniq!
					rep = convex_vertices(rep, last, nxt, h1, h2)
#rep.each{|v| gen_arc(v.x, v.y, 10, 0, 5, 5)}

#puts 'ccc', rep.index(last) 

					rep.rotate(rep.index(last)) if rep.index(last)

					rep.delete(last)
					rep.delete(cur)
					rep.delete(nxt)
					#rep.delete(h1)
					#rep.delete(h2)
					if (rep.length > 1) && ((rep[0].x - last.x) ** 2 + (rep[0].y - last.y) ** 2) > ((rep[-1].x - last.x) ** 2 + (rep[-1].y - last.y) ** 2)
						rep.reverse!
					end
					replace(cur, id, rep)
				end
			}
		}
		end
	end

	def draw_routes
		set_color(0, 0, 0, 1)
		@vertices.each{|vert|
			vert.incident_nets.each{|n|
				if false #n.next
					@pic.new_sub_path
					@pic.arc(vert.x, vert.y, 40, 0, Math::PI)
					@pic.stroke
				end
				last = vert
				lastx = lasty = nil
				lr = 0
				to = n.next
				to_net = last.net(n.id)
				while to do
					last_net = to_net
					fail if to_net.nstep != to.net(n.id)

				  to_net = to.net(n.id)
					
					radius = to_net.radius
					#puts 'rrrrrrr', radius
					t = get_tangents(last.x, last.y, lr, last_net.rgt, to.x, to.y, radius, to_net.rgt)
					gen_line(*t, 5)
					if lr > 0
					  start_angle = Math.atan2(lasty - last.y, lastx - last.x)
						end_angle = Math.atan2(t[1] - last.y, t[0] - last.x)
						start_angle, end_angle = end_angle, start_angle unless last_net.rgt
						gen_arc(last.x, last.y, lr, start_angle, end_angle, 5)
					end
					lr = radius
					last = to
					to = to_net.next
					lastx = t[2]
					lasty = t[3]
				end
			}
		}
	end

	def set_line_width(w)
		@pic.set_line_width(w)
	end
	def set_color(r, g, b, a)
		@pic.set_source_rgba(r, g, b, a)
	end
	def save_picture
		@image.write_to_png('pic.png')
	end
end

init_seed
r = Router.new
r.draw_vertices

col = ([1, 0, 0].permutation(3).to_a + [1, 1, 0].permutation(3).to_a).uniq - [[1, 0, 0]]
r.set_color(1, 0, 0, 0.7)
r.set_line_width(4)
(0..9).each{|i|
r.set_color(*col[i % 5], 0.7)
r.route(i)
}
r.sort_attached_nets
r.no_prepare_steps
r.nubly
r.draw_routes
r.save_picture


